<?php
// backup/upload_chunk.php
header("Content-Type: application/json");

require __DIR__ . "/../auth_check.php";
require __DIR__ . "/../core/storage/StorageFactory.php";
require __DIR__ . "/../core/services/NotificationService.php";
require __DIR__ . "/../core/security/DeviceIsolationService.php";

/* -------------------------------------------------
 * INPUT
 * ------------------------------------------------- */
$userId   = (int)$auth['uid'];
$deviceId = $auth['device_id'] ?? null;

$fileId      = $_POST['file_id'] ?? '';
$fileName    = basename($_POST['file_name'] ?? '');
$chunkIndex  = (int)($_POST['chunk_index'] ?? -1);
$totalChunks = (int)($_POST['total_chunks'] ?? 0);
$fileSize    = (int)($_POST['file_size'] ?? 0);

if (!$fileId || !$fileName || $chunkIndex < 0 || $totalChunks <= 0) {
    http_response_code(400);
    exit(json_encode(["error" => "Invalid chunk metadata"]));
}

if (!isset($_FILES['chunk'])) {
    http_response_code(400);
    exit(json_encode(["error" => "Missing chunk"]));
}

/* -------------------------------------------------
 * QUOTA + EXPIRY CHECK (FIRST CHUNK ONLY)
 * ------------------------------------------------- */
if ($chunkIndex === 0) {

    $stmt = $pdo->prepare("
        SELECT 
            (p.quota_gb * 1024 * 1024 * 1024) AS quota_bytes,
            IFNULL(up.used_bytes,0) AS used_bytes,
            up.expires_on
        FROM user_packages up
        JOIN packages p ON p.id = up.package_id
        WHERE up.user_id = ?
    ");
    $stmt->execute([$userId]);
    $pkg = $stmt->fetch(PDO::FETCH_ASSOC);

    if (!$pkg) {
        http_response_code(403);
        exit(json_encode(["error" => "No active package"]));
    }

    if (!empty($pkg['expires_on']) && strtotime($pkg['expires_on']) < time()) {
        http_response_code(403);
        exit(json_encode(["error" => "Subscription expired"]));
    }

    if ($pkg['used_bytes'] + $fileSize > $pkg['quota_bytes']) {
        http_response_code(403);
        exit(json_encode(["error" => "Quota exceeded"]));
    }
}

/* -------------------------------------------------
 * TEMP STORAGE
 * ------------------------------------------------- */
$baseDir = __DIR__ . "/../storage/tmp_chunks/$userId/$fileId";
if (!is_dir($baseDir)) {
    mkdir($baseDir, 0777, true);
}

move_uploaded_file(
    $_FILES['chunk']['tmp_name'],
    "$baseDir/$chunkIndex.part"
);

/* -------------------------------------------------
 * PARTIAL UPLOAD
 * ------------------------------------------------- */
$parts = glob("$baseDir/*.part");
if (count($parts) < $totalChunks) {
    exit(json_encode([
        "status" => "partial",
        "uploaded_chunks" => count($parts)
    ]));
}

/* -------------------------------------------------
 * ASSEMBLE FILE
 * ------------------------------------------------- */
$finalTmp = "$baseDir/complete.tmp";
$out = fopen($finalTmp, 'wb');

for ($i = 0; $i < $totalChunks; $i++) {
    $part = "$baseDir/$i.part";
    if (!file_exists($part)) {
        fclose($out);
        exit(json_encode(["error" => "Missing chunk $i"]));
    }
    fwrite($out, file_get_contents($part));
}
fclose($out);

/* -------------------------------------------------
 * RANSOMWARE HEURISTIC (ENTROPY)
 * ------------------------------------------------- */
function entropy(string $file): float {
    $data = file_get_contents($file, false, null, 0, 65536);
    if (!$data) return 0;
    $freq = count_chars($data, 1);
    $len  = strlen($data);
    $h = 0;
    foreach ($freq as $c) {
        $p = $c / $len;
        $h -= $p * log($p, 2);
    }
    return $h;
}

$entropyVal = entropy($finalTmp);

if ($entropyVal > 7.8 && $deviceId) {

    // 🔒 Isolate only this device
    DeviceIsolationService::block(
        $pdo,
        (int)$deviceId,
        'High-entropy chunked upload detected'
    );

    NotificationService::push($pdo, [
        'user_id'  => $userId,
        'title'    => 'Device Blocked',
        'message'  => 'A device was blocked due to suspicious backup activity.',
        'severity' => 'critical'
    ]);

    NotificationService::notifyAdmins(
        $pdo,
        'Ransomware Alert (Chunked)',
        "Device {$deviceId} isolated for user {$userId}"
    );

    http_response_code(403);
    exit(json_encode([
        "error" => "This device has been blocked for security reasons"
    ]));
}

/* -------------------------------------------------
 * VERSIONING
 * ------------------------------------------------- */
$virtualPath = $fileName;

$stmt = $pdo->prepare("
    SELECT IFNULL(MAX(version_no),0)
    FROM backup_files
    WHERE user_id = ? AND virtual_path = ?
");
$stmt->execute([$userId, $virtualPath]);
$versionNo = ((int)$stmt->fetchColumn()) + 1;

/* -------------------------------------------------
 * STORAGE UPLOAD
 * ------------------------------------------------- */
$servers = $pdo->query("
    SELECT * FROM backup_servers
    ORDER BY FIELD(type,'sftp','ftp','http','s3')
")->fetchAll();

$storage = null;
$activeServer = null;

foreach ($servers as $s) {
    try {
        $storage = getStorage($s);
        $storage->upload($finalTmp, "/$userId/$virtualPath");
        $activeServer = $s;
        break;
    } catch (Exception $e) {}
}

if (!$storage) {
    NotificationService::push($pdo, [
        'user_id'  => $userId,
        'title'    => 'Backup Failed',
        'message'  => 'Chunked upload failed.',
        'severity' => 'critical'
    ]);
    exit(json_encode(["error" => "No storage backend available"]));
}

/* -------------------------------------------------
 * DATABASE LOGGING
 * ------------------------------------------------- */
$pdo->prepare("
    INSERT INTO backup_files
        (user_id, server_id, path, virtual_path, upload_mode, size, version_no, created_at)
    VALUES
        (?, ?, ?, ?, 'sync', ?, ?, NOW())
")->execute([
    $userId,
    $activeServer['id'],
    "/$userId/$virtualPath",
    $virtualPath,
    $fileSize,
    $versionNo
]);

$pdo->prepare("
    UPDATE user_packages
    SET used_bytes = used_bytes + ?
    WHERE user_id = ?
")->execute([$fileSize, $userId]);

/* -------------------------------------------------
 * CLEANUP
 * ------------------------------------------------- */
array_map('unlink', glob("$baseDir/*.part"));
@unlink($finalTmp);
@rmdir($baseDir);
@rmdir(dirname($baseDir));

echo json_encode([
    "status"  => "completed",
    "file"    => $fileName,
    "version" => $versionNo
]);
